// اطلاعات کاربر
const username = localStorage.getItem('voicechat_username') || 'کاربر';
const userId = 'user_' + Date.now();

// متغیرها
let socket = null;
let isMicOn = true;
let isTalking = false;
let myAudio = null;
let audioContext = null;
let analyser = null;

// وقتی صفحه لود شد
window.onload = function() {
    console.log('اتاق صوتی بارگذاری شد - کاربر:', username);
    
    // راه‌اندازی صدا
    setupAudio();
    
    // اتصال به سرور
    connectWebSocket();
    
    // به‌روزرسانی لیست کاربران
    setInterval(updateUserList, 3000);
    
    // آنالیز صدای میکروفون
    setInterval(updateSoundLevel, 100);
    
    // رویدادهای کیبورد
    document.addEventListener('keydown', function(e) {
        if (e.code === 'Space') {
            e.preventDefault();
            startTalking();
        }
    });
    
    document.addEventListener('keyup', function(e) {
        if (e.code === 'Space') {
            stopTalking();
        }
    });
};

// راه‌اندازی میکروفون
async function setupAudio() {
    try {
        const stream = await navigator.mediaDevices.getUserMedia({
            audio: {
                echoCancellation: true,
                noiseSuppression: true
            }
        });
        
        myAudio = document.getElementById('my-audio');
        myAudio.srcObject = stream;
        
        // برای آنالیز صدا
        audioContext = new (window.AudioContext || window.webkitAudioContext)();
        const source = audioContext.createMediaStreamSource(stream);
        analyser = audioContext.createAnalyser();
        analyser.fftSize = 256;
        source.connect(analyser);
        
        console.log('میکروفون آماده است');
        
    } catch(error) {
        console.error('خطا در میکروفون:', error);
        alert('میکروفون پیدا نشد!');
    }
}

// اتصال به WebSocket
function connectWebSocket() {
    const wsUrl = 'ws://localhost:8080';
    
    socket = new WebSocket(wsUrl);
    
    socket.onopen = function() {
        console.log('متصل به سرور');
        
        // معرفی خودم
        socket.send(JSON.stringify({
            type: 'join',
            userId: userId,
            username: username
        }));
    };
    
    socket.onmessage = function(event) {
        const data = JSON.parse(event.data);
        handleMessage(data);
    };
    
    socket.onclose = function() {
        console.log('اتصال قطع شد');
        setTimeout(connectWebSocket, 3000);
    };
    
    socket.onerror = function(error) {
        console.error('خطای WebSocket:', error);
    };
}

// مدیریت پیام‌ها
function handleMessage(data) {
    switch(data.type) {
        case 'users':
            showUsers(data.users);
            break;
            
        case 'user_joined':
            addUser(data.user);
            break;
            
        case 'user_left':
            removeUser(data.userId);
            break;
            
        case 'talking':
            userTalking(data.userId, true);
            break;
            
        case 'stopped_talking':
            userTalking(data.userId, false);
            break;
            
        case 'chat':
            addChatMessage(data.username, data.message);
            break;
    }
}

// نمایش کاربران
function showUsers(users) {
    const container = document.getElementById('online-list');
    container.innerHTML = '';
    
    // کاربران دیگر
    users.forEach(user => {
        if(user.userId !== userId) {
            const div = document.createElement('div');
            div.className = 'user-tag';
            div.id = 'user-' + user.userId;
            div.innerHTML = `<i class="fas fa-user"></i> ${user.username}`;
            container.appendChild(div);
        }
    });
    
    // خودم
    const meDiv = document.createElement('div');
    meDiv.className = 'user-tag';
    meDiv.innerHTML = `<i class="fas fa-user"></i> ${username} (من)`;
    container.appendChild(meDiv);
}

// اضافه کردن کاربر جدید
function addUser(user) {
    const container = document.getElementById('online-list');
    
    const div = document.createElement('div');
    div.className = 'user-tag';
    div.id = 'user-' + user.userId;
    div.innerHTML = `<i class="fas fa-user"></i> ${user.username}`;
    
    container.insertBefore(div, container.lastChild);
    
    // اطلاع در چت
    addChatMessage('سیستم', `${user.username} وارد اتاق شد`);
}

// حذف کاربر
function removeUser(userId) {
    const userEl = document.getElementById('user-' + userId);
    if(userEl) {
        userEl.remove();
    }
}

// شروع صحبت
function startTalking() {
    if(!isMicOn) return;
    
    isTalking = true;
    document.getElementById('talk-btn').style.opacity = '0.8';
    
    // اطلاع به سرور
    if(socket) {
        socket.send(JSON.stringify({
            type: 'talking',
            userId: userId,
            username: username
        }));
    }
}

// توقف صحبت
function stopTalking() {
    if(!isTalking) return;
    
    isTalking = false;
    document.getElementById('talk-btn').style.opacity = '1';
    
    if(socket) {
        socket.send(JSON.stringify({
            type: 'stopped_talking',
            userId: userId
        }));
    }
}

// کاربر دیگر در حال صحبت
function userTalking(userId, talking) {
    const userEl = document.getElementById('user-' + userId);
    if(userEl) {
        if(talking) {
            userEl.classList.add('talking');
            document.getElementById('talking-status').textContent = 'کسی در حال صحبت است...';
        } else {
            userEl.classList.remove('talking');
            document.getElementById('talking-status').textContent = 'آماده برای صحبت...';
        }
    }
}

// خاموش/روشن میکروفون
function toggleMic() {
    isMicOn = !isMicOn;
    const btn = document.getElementById('mic-btn');
    
    if(isMicOn) {
        btn.className = 'mic-on';
        btn.innerHTML = '<i class="fas fa-microphone"></i> میکروفون روشن';
    } else {
        btn.className = 'mic-off';
        btn.innerHTML = '<i class="fas fa-microphone-slash"></i> میکروفون خاموش';
    }
}

// ارسال پیام متنی
function sendMessage() {
    const input = document.getElementById('message-input');
    const message = input.value.trim();
    
    if(!message || !socket) return;
    
    socket.send(JSON.stringify({
        type: 'chat',
        userId: userId,
        username: username,
        message: message
    }));
    
    input.value = '';
}

// اضافه کردن پیام به چت
function addChatMessage(sender, text) {
    const container = document.getElementById('chat-messages');
    const div = document.createElement('div');
    div.className = 'message';
    div.innerHTML = `<span class="name">${sender}:</span> ${text}`;
    
    container.appendChild(div);
    container.scrollTop = container.scrollHeight;
}

// به‌روزرسانی لیست کاربران
function updateUserList() {
    if(socket && socket.readyState === WebSocket.OPEN) {
        socket.send(JSON.stringify({ type: 'get_users' }));
    }
}

// نمایش سطح صدا
function updateSoundLevel() {
    if(!analyser || !isTalking) {
        document.getElementById('sound-level').style.width = '0%';
        return;
    }
    
    const data = new Uint8Array(analyser.frequencyBinCount);
    analyser.getByteFrequencyData(data);
    
    let sum = 0;
    for(let i = 0; i < data.length; i++) {
        sum += data[i];
    }
    
    const average = sum / data.length;
    const level = Math.min(100, (average / 255) * 200);
    
    document.getElementById('sound-level').style.width = level + '%';
}

// خروج از اتاق
function exitRoom() {
    if(socket) {
        socket.send(JSON.stringify({
            type: 'leave',
            userId: userId
        }));
        socket.close();
    }
    
    localStorage.removeItem('voicechat_username');
    window.location.href = 'index.php';
}